﻿using Autodesk.AutoCAD.ApplicationServices;
using Autodesk.AutoCAD.EditorInput;
using Autodesk.AutoCAD.Runtime;
using Autodesk.Windows;
using System.Runtime.Serialization;
using System.Collections.Generic;
using System.Windows.Documents;
using System.Windows.Controls;
using System.Windows;
using System.Text;
using System.Linq;
using System.Net;
using System.Xml;
using System.IO;
using System;

[assembly: ExtensionApplication(typeof(TranslateTooltips.Commands))]

namespace TranslateTooltips
{
  public class Commands : IExtensionApplication
  {
    // Keep track of currently translated items

    static List<string> _handled = null;
    
    // Our source and target languages

    static string _srcLang = "en";
    static string _trgLang = "";

    public void Initialize()
    {
      DemandLoading.RegistryUpdate.RegisterForDemandLoading();
      HijackTooltips();
    }

    public void Terminate()
    {
    }

    [CommandMethod("ADNPLUGINS", "TRANSTIPSSRC", CommandFlags.Modal)]
    public static void ChooseSourceLanguage()
    {
      Document doc =
        Autodesk.AutoCAD.ApplicationServices.Application.
          DocumentManager.MdiActiveDocument;
      Editor ed = doc.Editor;

      // Get the list of language codes and their corresponding
      // names

      List<string> codes = GetLanguageCodes();
      string[] names = GetLanguageNames(codes);

      // Make sure we have as many names as languages supported

      if (codes.Count == names.Length)
      {
        // Ask the user to select a source language

        string lang =
          ChooseLanguage(ed, codes, names, _srcLang, true);

        // If the language code returned is neither empty
        // nor the same as the target, print the code's
        // name and set it as the new target

        if (lang == _trgLang)
        {
          ed.WriteMessage(
            "\nSource language cannot be the same as the " +
            "target language."
          );
        }
        else if (!String.IsNullOrEmpty(lang))
        {
          // Get the name corresponding to a language code

          string name =
            names[
              codes.FindIndex(0, x => x == lang)
            ];

          // Print it to the user

          ed.WriteMessage(
            "\nSource language set to {0}.\n", name
          );

          // Set the new source language

          _srcLang = lang;
        }
      }
    }

    [CommandMethod("ADNPLUGINS", "TRANSTIPS", CommandFlags.Modal)]
    public static void ChooseTranslationLanguage()
    {
      Document doc =
        Autodesk.AutoCAD.ApplicationServices.Application.
          DocumentManager.MdiActiveDocument;
      Editor ed = doc.Editor;

      // Get the list of language codes and their corresponding
      // names

      List<string> codes = GetLanguageCodes();
      string[] names = GetLanguageNames(codes);

      // Make sure we have as many names as languages supported

      if (codes.Count == names.Length)
      {
        // Ask the user to select a target language

        string lang =
          ChooseLanguage(ed, codes, names, _trgLang, false);

        // If the language code returned is empty or the same
        // as the source, turn off translations

        if (lang == "" || lang == _srcLang)
        {
          ed.WriteMessage(
            "\nTooltip translation is turned off."
          );
          _trgLang = "";
        }
        else if (lang != null)
        {
          // Otherwise get the name corresponding to a language
          // code

          string name =
            names[
              codes.FindIndex(0, x => x == lang)
            ];

          // Print it to the user

          ed.WriteMessage(
            "\nTooltips will be translated into {0}.\n", name
          );

          // Set the new target language

          _trgLang = lang;
        }
      }
    }

    private static string ChooseLanguage(
      Editor ed, List<string> codes, string[] names,
      string lang, bool source
    )
    {
      // First option (0) is to unselect

      ed.WriteMessage("\n0 None");

      // The others (1..n) are the languages
      // available on the server

      for (int i = 0; i < names.Length; i++)
      {
        ed.WriteMessage("\n{0} {1}", i + 1, names[i]);
      }

      // Adjust the prompt based on whether selecting
      // a source or target language

      PromptIntegerOptions pio =
        new PromptIntegerOptions(
          String.Format(
            "\nEnter number of {0} language to select: ",
            source ? "source" : "target"
          )
        );

      // Add each of the codes as hidden keywords, which
      // allows the user to also select the language using
      // the 2-digit code (good for scripting on startup,
      // to avoid having to hard code a number)

      foreach (string code in codes)
      {
        pio.Keywords.Add(code, code, code, false, true);
      }

      // Set the bounds and the default value

      pio.LowerLimit = 0;
      pio.UpperLimit = names.Length;
      if (codes.Contains(lang))
      {
        pio.DefaultValue =
          codes.FindIndex(0, x => lang == x) + 1;
      }
      else
      {
        pio.DefaultValue = 0;
      }
      pio.UseDefaultValue = true;

      // Get the selection

      PromptIntegerResult pir = ed.GetInteger(pio);

      string resLang = null;

      if (pir.Status == PromptStatus.Keyword)
      {
        // The code was entered as a string

        if (!codes.Contains(pir.StringResult))
        {
          ed.WriteMessage(
            "\nNot a valid language code."
          );
          resLang = null;
        }
        else
        {
          resLang = pir.StringResult;
        }
      }
      else if (pir.Status == PromptStatus.OK)
      {
        // A number was selected

        if (pir.Value == 0)
        {
          // A blank string indicates none

          resLang = "";
        }
        else
        {
          // Otherwise we return the corresponding
          // code

          resLang = codes[pir.Value - 1];
        }
      }

      return resLang;
    }

    public static void HijackTooltips()
    {
      Document doc =
        Autodesk.AutoCAD.ApplicationServices.Application.
          DocumentManager.MdiActiveDocument;
      Editor ed = doc.Editor;

      _handled = new List<string>();

      // Respond to an event fired when any tooltip is
      // displayed inside AutoCAD

      Autodesk.Windows.ComponentManager.ToolTipOpened +=
        (s, e) =>
        {
          if (!String.IsNullOrEmpty(_trgLang))
          {
            // The outer object is of an Autodesk.Internal
            // class, hence subject to change

            Autodesk.Internal.Windows.ToolTip tt =
              s as Autodesk.Internal.Windows.ToolTip;
            if (tt != null)
            {
              if (tt.Content is RibbonToolTip)
              {
                // Enhanced tooltips

                RibbonToolTip rtt = (RibbonToolTip)tt.Content;

                rtt.Content =
                  TranslateIfString(
                    rtt.Content, rtt.Command
                  );
                TranslateObjectContent(
                  rtt.Content, rtt.Command
                );

                // Translate any expanded content
                // (adding a suffix to the ID to
                // distinguish from the basic content)

                rtt.ExpandedContent =
                  TranslateIfString(
                    rtt.ExpandedContent, rtt.Command + "-x"
                  );
                TranslateObjectContent(
                  rtt.ExpandedContent, rtt.Command + "-x"
                );
              }
              else if (tt.Content is UriKey)
              {
                // This is called once for tooltips that
                // need to be resolved by the system
                
                // Here we close the current tooltip and
                // move the cursor to 0,0 and back again,
                // to cause the resolved tooltip to be
                // displayed, which will call this event
                // again following a different path
                
                tt.Close();
                System.Drawing.Point pt =
                  System.Windows.Forms.Cursor.Position;
                System.Windows.Forms.Cursor.Position =
                  System.Drawing.Point.Empty;
                System.Windows.Forms.Application.DoEvents();
                System.Windows.Forms.Cursor.Position = pt;
              }
              else
              {
                // A basic, string-only tooltip

                tt.Content = TranslateIfString(tt.Content, null);
              }
            }
          }
        };
    }

    private static object TranslateIfString(
      object obj, string id
    )
    {
      // If the object passed in is a string,
      // return its translation to the caller

      object ret = obj;
      if (obj is string)
      {
        string trans =
          TranslateContent((string)obj, id);
        if (!String.IsNullOrEmpty(trans))
        {
          ret = trans;
          MarkAsTranslated(id);
        }
      }
      return ret;
    }

    private static void TranslateObjectContent(
      object obj, string id
    )
    {
      // Translate more complex objects and their
      // contents

      if (obj != null)
      {
        if (obj is TextBlock)
        {
          // Translate TextBlocks

          TextBlock tb = (TextBlock)obj;
          TranslateTextBlock(tb, id);
        }
        else if (obj is StackPanel)
        {
          // And also handle StackPanels of content

          StackPanel sp = (StackPanel)obj;
          TranslateStackPanel(sp, id);
        }
      }
    }

    private static void TranslateTextBlock(
      TextBlock tb, string id
    )
    {
      // Translate a TextBlock

      string trans =
        TranslateContent(tb.Text, id);
      if (!String.IsNullOrEmpty(trans))
      {
        tb.Text = trans;
        MarkAsTranslated(id);
      }
    }

    private static void TranslateStackPanel(
      StackPanel sp, string id
    )
    {
      // Translate a StackPanel of content

      TextBlock tb;
      foreach (UIElement elem in sp.Children)
      {
        tb = elem as TextBlock;
        if (tb != null)
        {
          TranslateTextBlock(tb, id);
        }
        else
        {
          FlowDocumentScrollViewer sv =
            elem as FlowDocumentScrollViewer;
          if (sv != null)
          {
            TranslateFlowDocumentScrollViewer(
              sv, id
            );
          }
        }
      }
    }

    private static void TranslateFlowDocumentScrollViewer(
      FlowDocumentScrollViewer sv, string id
    )
    {
      // Translate a FlowDocumentScrollViewer, which
      // hosts content such as bullet-lists in
      // certain tooltips (e.g. for HATCH)

      int n = 0;
      Block b = sv.Document.Blocks.FirstBlock;
      while (b != null)
      {
        List l = b as List;
        if (l != null)
        {
          ListItem li = l.ListItems.FirstListItem;
          while (li != null)
          {
            Block b2 = li.Blocks.FirstBlock;
            while (b2 != null)
            {
              Paragraph p = b2 as Paragraph;
              if (p != null)
              {
                Inline i = p.Inlines.FirstInline;
                while (i != null)
                {
                  string contents =
                    i.ContentStart.GetTextInRun(
                      LogicalDirection.Forward
                    );

                  // We need to suffix the IDs to
                  // keep them distinct

                  string trans =
                    TranslateContent(
                      contents, id + n.ToString()
                    );
                  if (!String.IsNullOrEmpty(trans))
                  {
                    i.ContentStart.DeleteTextInRun(
                      contents.Length
                    );
                    i.ContentStart.InsertTextInRun(trans);
                    MarkAsTranslated(id + n.ToString());
                  }
                  n++;
                  i = i.NextInline;
                }
              }
              b2 = b2.NextBlock;
            }
            li = li.NextListItem;
          }
        }
        b = b.NextBlock;
      }
    }

    private static void MarkAsTranslated(string id)
    {
      // Mark an item as having been translated

      if (!String.IsNullOrEmpty(id) && !_handled.Contains(id))
        _handled.Add(id);
    }

    private static void UnmarkAsTranslated(string id)
    {
      // Remove an item from the list of marked items

      if (!String.IsNullOrEmpty(id) && _handled.Contains(id))
        _handled.Remove(id);
    }

    private static bool AlreadyTranslated(string id)
    {
      // Check the list, to see whether an item has been
      // translated

      return _handled.Contains(id);
    }

    private static string TranslateContent(
      string contents, string id
    )
    {
      // Our translation to return, and the string to pass
      // as source (which may be the string passed in,
      // or may come from disk)

      string trans = null,
             source = contents;

      // If the target language is empty, we'll take the source
      // language

      string trgLang =
        String.IsNullOrEmpty(_trgLang) ? _srcLang : _trgLang;

      // Get the name of the XML file for this data

      string fn = GetXmlFileName(id);
      if (File.Exists(fn))
      {
        // If the item has already been translated in the UI,
        // we need to load the source language version to
        // retranslate that

        if (AlreadyTranslated(id))
        {
          source = LoadXmlTranslation(fn, _srcLang);
          source = (source == null ? contents : source);

          // If the source and target are the same,
          // reset to the source language
          
          if (_srcLang == trgLang)
          {
            UnmarkAsTranslated(id);
            return source;
          }
        }

        // Attempt to get a prior translation from XML

        trans = LoadXmlTranslation(fn, trgLang);
      }

      if (trans == null)
      {
        // If there was no translation on disk, translate
        // via the online service

        trans =
          GetTranslatedText(_srcLang, trgLang, source);

        // If the filename is valid, save the data to XML

        if (!String.IsNullOrEmpty(fn))
        {
          SaveXmlTranslation(
            fn, _srcLang, source, trgLang, trans
          );
        }
      }

      return trans;
    }

    private static string GetXmlFileName(string id)
    {
      // The XML file will be beneath My Documents, under
      // a sub-folder named "TransTip Cache"

      string path =
        System.Environment.GetFolderPath(
          Environment.SpecialFolder.MyDocuments
        ) + "\\TransTips Cache";

      if (!Directory.Exists(path))
      {
        Directory.CreateDirectory(path);
      }

      // The filenae is the id with the .xml extension

      return id == null ? "" : path + "\\" + id + ".xml";
    }

    private static void SaveXmlTranslation(
      string fn,
      string srcLang, string contents,
      string trgLang, string trans
    )
    {
      if (File.Exists(fn))
      {
        // Add our content to an existing XML file

        XmlDocument xd = new XmlDocument();
        xd.Load(fn);
        XmlNode tg =
          xd.SelectSingleNode(
            "/Translation/Content[@Language='" + trgLang + "']"
          );
        if (tg == null)
        {
          XmlNode xn =
            xd.CreateNode(XmlNodeType.Element, "Content", "");
          XmlAttribute xlang = xd.CreateAttribute("Language");
          xlang.Value = trgLang;
          xn.InnerText = trans;
          xn.Attributes.Append(xlang);
          xd.GetElementsByTagName("Translation")[0].InsertAfter(
            xn,
            xd.GetElementsByTagName("Translation")[0].LastChild
          );
          xd.Save(fn);
        }
      }
      else
      {
        // Create a new Unicode XML file

        XmlTextWriter xw =
          new XmlTextWriter(fn, Encoding.Unicode);
        using (xw)
        {
          xw.WriteStartDocument();
          xw.WriteStartElement("Translation");
          xw.WriteStartElement("Content");
          xw.WriteAttributeString("Language", srcLang);
          xw.WriteAttributeString("Source", "True");
          xw.WriteString(contents);
          xw.WriteEndElement();
          xw.WriteStartElement("Content");
          xw.WriteAttributeString("Language", trgLang);
          xw.WriteString(trans);
          xw.WriteEndElement();
          xw.WriteEndElement();
          xw.Close();
        }
      }
    }

    private static string LoadXmlTranslation(
      string fn, string lang
    )
    {
      // Load the XML document

      XmlDocument xd = new XmlDocument();
      xd.Load(fn);
      
      // Look for a Content node for our language
      
      XmlNode tg =
        xd.SelectSingleNode(
          "/Translation/Content[@Language='" + lang + "']"
        );
      return tg == null ? null : tg.InnerXml;
    }

    // Replace the following string with the AppId you receive
    // from the Bing Developer Center
    
    const string AppId =
      "5167D5D514A6618A82D2965BED70CF692389EE84";

    private static string GetTranslatedText(
      string from, string to, string content
    )
    {
      // Translate a string from one language to another

      string uri =
        "http://api.microsofttranslator.com/v2/Http.svc/" +
        "Translate?appId=" + AppId + "&text=" + content +
        "&from=" + from + "&to=" + to;

      // Create the request

      HttpWebRequest request =
        (HttpWebRequest)WebRequest.Create(uri);

      string output = null;
      WebResponse response = null;

      try
      {
        // Get the response

        response = request.GetResponse();
        Stream strm = response.GetResponseStream();

        // Extract the results string

        DataContractSerializer dcs =
          new DataContractSerializer(
            Type.GetType("System.String")
          );
        output = (string)dcs.ReadObject(strm);
      }
      catch (WebException e)
      {
        ProcessWebException(
          e, "\nFailed to translate text."
        );
      }
      finally
      {
        if (response != null)
        {
          response.Close();
          response = null;
        }
      }
      return output;
    }

    private static List<string> GetLanguageCodes()
    {
      // Get the list of language codes supported

      string uri =
        "http://api.microsofttranslator.com/v2/Http.svc/" +
        "GetLanguagesForTranslate?appId=" + AppId;

      // Create the request

      HttpWebRequest request =
        (HttpWebRequest)WebRequest.Create(uri);

      WebResponse response = null;
      List<String> codes = null;

      try
      {
        // Get the response

        response = request.GetResponse();
        using (Stream stream = response.GetResponseStream())
        {
          // Extract the list of language codes

          DataContractSerializer dcs =
            new DataContractSerializer(typeof(List<String>));

          codes = (List<String>)dcs.ReadObject(stream);
        }
      }
      catch (WebException e)
      {
        ProcessWebException(
          e, "\nFailed to get target translation languages."
        );
      }
      finally
      {
        if (response != null)
        {
          response.Close();
          response = null;
        }
      }
      return codes;
    }

    public static string[] GetLanguageNames(List<string> codes)
    {
      string uri =
        "http://api.microsofttranslator.com/v2/Http.svc/" +
        "GetLanguageNames?appId=" + AppId + "&locale=en";

      // Create the request
      
      HttpWebRequest req =
        (HttpWebRequest)WebRequest.Create(uri);
            
      req.ContentType = "text/xml";
      req.Method = "POST";

      // Encode the list of language codes

      DataContractSerializer dcs =
        new DataContractSerializer(
          Type.GetType("System.String[]")
        );
      using (Stream stream = req.GetRequestStream())
      {
        dcs.WriteObject(stream, codes.ToArray());
      }

      WebResponse response = null;
      try
      {
        // Get the response

        response = req.GetResponse();

        using (Stream stream = response.GetResponseStream())
        {
          // Extract the list of language names

          string[] results = (string[])dcs.ReadObject(stream);
          string[] names =
            results.Select(x => x.ToString()).ToArray();
          return names;
        }
      }
      catch (WebException e)
      {
        ProcessWebException(
          e, "\nFailed to get target language."
        );
      }
      finally
      {
        if (response != null)
        {
          response.Close();
          response = null;
        }
      }
      return null;
    }

    private static void ProcessWebException(
      WebException e, string message
    )
    {
      // Provide information regarding an exception

      Document doc =
        Autodesk.AutoCAD.ApplicationServices.Application.
          DocumentManager.MdiActiveDocument;
      Editor ed = doc.Editor;

      ed.WriteMessage("{0}: {1}", message, e.ToString());

      // Obtain detailed error information

      string strResponse = string.Empty;
      using (
        HttpWebResponse response =
          (HttpWebResponse)e.Response
      )
      {
        using (
          Stream responseStream =
            response.GetResponseStream()
        )
        {
          using (
            StreamReader sr =
              new StreamReader(
                responseStream, System.Text.Encoding.ASCII
              )
            )
          {
            strResponse = sr.ReadToEnd();
          }
        }
      }

      // Print it to the user

      ed.WriteMessage(
        "\nHttp status code={0}, error message={1}",
        e.Status, strResponse
      );
    }                               
  }
}
